#include "util.h"
#include <avr/io.h>
#include <util/delay.h>

#include "hd44780.h"

#define LCD_CTRL_IN PINC
#define LCD_CTRL_OUT PORTC
#define LCD_CTRL_DDR DDRC
#define LCD_CTRL_RS 1
#define LCD_CTRL_RW 2
#define LCD_CTRL_E  4

#define LCD_DATA_DDR DDRB
#define LCD_DATA_IN PINB
#define LCD_DATA_OUT PORTB

#define LCD_I2C_ADDR 0x20


static void i2c_init()
{
   TWSR = 0x00;
   TWBR = ( F_CPU / 100000 ) - 8;
   TWCR = ( 1 << TWEN );
}


static void i2c_stop()
{
   TWCR = ( 1 << TWINT ) | ( 1 << TWSTO ) | ( 1 << TWEN );
}


static void i2c_write( uint8_t d )
{
   TWDR = d;
   TWCR = ( 1 << TWINT ) | ( 1 << TWEN );
   while( ( TWCR & ( 1 << TWINT ) ) == 0 );
}


static void i2c_start()
{
   TWCR = ( 1 << TWINT ) | ( 1 << TWSTA ) | ( 1 << TWEN );
   while( ( TWCR & ( 1 << TWINT ) ) == 0 );
   
//   i2c_write( ( deviceAddr << 1 ) | ( rw ? 1 : 0 ) );
}
/*
// 9)
void lcd_waitbusy()
{
   // Read
   input( LCD_DATA_DDR, 0xff );
   
   while( 1 )
   {
      clearPins( LCD_CTRL_OUT, LCD_CTRL_E );
      _delay_us( 1 );
      setPins( LCD_CTRL_OUT, LCD_CTRL_RW );
      clearPins( LCD_CTRL_OUT, LCD_CTRL_RS );
      _delay_us( 1 );
      setPins(   LCD_CTRL_OUT, LCD_CTRL_E );
      _delay_us( 1 );
      uint8_t busy = LCD_DATA_IN & 0x80;
      clearPins( LCD_CTRL_OUT, LCD_CTRL_E );
      _delay_us( 1 );
      
      if( !busy )
         break;
   }
}
*/

static void lcd_i2c_write_byte( uint8_t addr, uint8_t d )
{
   i2c_start();
   i2c_write( addr << 1 ); // bits 7..1 are the actual device ID,
                           // bit 0 is the read (=1) / write (=0) flag
   i2c_write( 1 );         // TCA9534 register 1: output port
   i2c_write( d );
   i2c_stop();
}


static void lcd_write( uint8_t d )
{
   lcd_i2c_write_byte( LCD_I2C_ADDR, d );
   lcd_i2c_write_byte( LCD_I2C_ADDR, d | LCD_CTRL_E );
   lcd_i2c_write_byte( LCD_I2C_ADDR, d );
}


static void lcd_set( int rs, uint8_t d, int upperNibbleOnly )
{
   uint8_t b1 = ( d & 0xf0 ) |            // The upper data nibble
                ( 1 << 3 )   |            // Switch on backlight
                ( rs ? LCD_CTRL_RS : 0 ); // Register select

   uint8_t b2 = ( ( d << 4 ) & 0xf0 ) |   // The lower data nibble
                ( 1 << 3 )            |   // Switch on backlight
                ( rs ? LCD_CTRL_RS : 0 ); // Register select (0 = command, 1 = data)

   // Upper nibble
   lcd_write( b1 );

   if( !upperNibbleOnly )
   {
      // Lower nibble
      lcd_write( b2 );
   }
}


void lcd_initialize()
{
    lcd_set( 0, 0x30, 1 );
    _delay_us( 4200 );
    lcd_set( 0, 0x30, 1 );
    _delay_us( 150 );
    lcd_set( 0, 0x30, 1 );
    _delay_us( 37 );
    lcd_set( 0, 0x20, 1 ); // Function Set - 4 bits mode
    _delay_us( 37 );
}


// 8)
void lcd_set_ddram_address( uint8_t a )
{
   lcd_set( 0, 0x80 | ( a & 0x7f ), 0 );
}


// 6)
// datLength: 0 - 4bit
//            1 - 8bit
// nLines:    0 - 1 line
//            1 - 2 lines
// nFont:     0 - 5x8 dots
//            1 - 5x10 dots
void lcd_set_function( int dataLength, int nLines, int nFont )
{
   lcd_set( 0,
             0x20 |
      ( dataLength ? 0x10 : 0x00 ) |
      ( nLines     ? 0x08 : 0x00 ) |
      ( nFont      ? 0x04 : 0x00 ),
      0 );
}


// 4)
// enableDisplay: 0 - Switch off entire display
//                1 - Switch on entire display
// enableCursor:  0 - Switch off cursor
//                1 - Switch on cursor
void lcd_ctrl_display_cursor( int enableDisplay, int enableCursor, int enableCursorBlinking )
{
   lcd_set( 0, 
            0x08 |
      ( enableDisplay        ? 0x04 : 0x00 ) |
      ( enableCursor         ? 0x02 : 0x00 ) |
      ( enableCursorBlinking ? 0x01 : 0x00 ),
      0
   );
}


// 2)
void lcd_return_home()
{
   lcd_set( 0, 0x02, 0 );
}


// 1)
void lcd_clear_display()
{
   lcd_set( 0, 0x01, 0 );
}


// 3)
// bIncrement:    0 - Decrement cursor on DDRAM write
//                1 - Increment cursor on DDRAM write
// bDisplayShift: 0 - Disable display shift
//                1 - Enable display shift
void lcd_set_entry_mode( int bIncrement, int bDisplayShift )
{
   lcd_set( 0,
             0x04 |
      ( bIncrement     ? 0x02 : 0x00 ) |
      ( bDisplayShift  ? 0x01 : 0x00 ),
      0
   );
}


void lcd_write_ram( uint8_t d )
{
   lcd_set( 1, d, 0 );
}


void lcd_init()
{
   i2c_init();

   // TCA9534 Register 3: Configuration
   i2c_start();
   i2c_write( LCD_I2C_ADDR << 1 );
   i2c_write( 3 );
   i2c_write( 0 ); // All output
   i2c_stop();
   
   _delay_us( 1000 );

   // TCA9534 Register 2: Polarity Inversion
   i2c_start();
   i2c_write( LCD_I2C_ADDR << 1 );
   i2c_write( 2 );
   i2c_write( 0 ); // No inversion
   i2c_stop();
   
   _delay_us( 1000 );

   // TCA9534 Register 1: Output port
   i2c_start();
   i2c_write( LCD_I2C_ADDR << 1 );
   i2c_write( 1 );
   i2c_write( 0 ); // Set all 8 output lines to 0
   i2c_stop();

   _delay_us( 1000 );

   // Set LCD to 4bit mode    
   lcd_initialize();

   lcd_set_function( 0, 1, 0 );
}

